--[[
  RobloxBackend Analytics SDK
  Version: 1.0.0
  
  Usage:
    local Analytics = require(game.ReplicatedStorage.AnalyticsSDK)
    Analytics.init("YOUR_API_KEY", "https://dogan.works")
    
    -- Track events
    Analytics.track(player.UserId, "session_start")
    Analytics.track(player.UserId, "item_purchased", { price = 100, itemId = "sword" })
    
    -- Batch track (more efficient)
    Analytics.batchTrack({
        { userId = player.UserId, eventName = "level_complete", properties = { level = 5 } }
    })
]]

local HttpService = game:GetService("HttpService")

local Analytics = {}
Analytics.__index = Analytics

local API_KEY = ""
local BASE_URL = "https://dogan.works"
local eventQueue = {}
local BATCH_SIZE = 50
local BATCH_INTERVAL = 5 -- seconds

function Analytics.init(apiKey, baseUrl)
    API_KEY = apiKey
    BASE_URL = baseUrl or "https://dogan.works"
    
    -- Start batch processor
    spawn(function()
        while true do
            wait(BATCH_INTERVAL)
            Analytics.flushQueue()
        end
    end)
end

function Analytics.track(userId, eventName, properties)
    table.insert(eventQueue, {
        userId = tostring(userId),
        eventName = eventName,
        properties = properties or {},
        timestamp = os.time() * 1000
    })
    
    if #eventQueue >= BATCH_SIZE then
        Analytics.flushQueue()
    end
    
    return true
end

function Analytics.flushQueue()
    if #eventQueue == 0 then return end
    
    local events = {}
    for i = 1, math.min(#eventQueue, 100) do
        table.insert(events, table.remove(eventQueue, 1))
    end
    
    spawn(function()
        local success, result = pcall(function()
            return HttpService:RequestAsync({
                Url = BASE_URL .. "/api/v1/analytics/batch",
                Method = "POST",
                Headers = {
                    ["Content-Type"] = "application/json",
                    ["x-api-key"] = API_KEY
                },
                Body = HttpService:JSONEncode({
                    events = events
                })
            })
        end)
        
        if not success or not result.Success then
            warn("[Analytics] Failed to send batch:", result)
            -- Re-queue failed events
            for _, event in ipairs(events) do
                table.insert(eventQueue, 1, event)
            end
        end
    end)
end

function Analytics.batchTrack(events)
    for _, event in ipairs(events) do
        event.userId = tostring(event.userId)
        event.timestamp = event.timestamp or (os.time() * 1000)
        event.properties = event.properties or {}
        table.insert(eventQueue, event)
    end
    
    if #eventQueue >= BATCH_SIZE then
        Analytics.flushQueue()
    end
    
    return true
end

function Analytics.trackImmediate(userId, eventName, properties)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/analytics/event",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                userId = tostring(userId),
                eventName = eventName,
                properties = properties or {},
                timestamp = os.time() * 1000
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.eventId
    else
        warn("[Analytics] Failed to track event:", result)
        return false, nil
    end
end

-- Common event helpers
function Analytics.sessionStart(userId)
    return Analytics.track(userId, "session_start")
end

function Analytics.firstSession(userId)
    return Analytics.track(userId, "first_session")
end

function Analytics.sessionEnd(userId, duration)
    return Analytics.track(userId, "session_end", { duration = duration })
end

function Analytics.purchase(userId, itemId, price, currency)
    return Analytics.track(userId, "item_purchased", {
        itemId = itemId,
        price = price,
        currency = currency or "ROBUX"
    })
end

function Analytics.levelComplete(userId, level, score)
    return Analytics.track(userId, "level_complete", {
        level = level,
        score = score
    })
end

function Analytics.tutorialComplete(userId)
    return Analytics.track(userId, "tutorial_complete")
end

function Analytics.getPlayerStats(userId)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/analytics/player/" .. tostring(userId),
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)
    
    if success and result.Success then
        return HttpService:JSONDecode(result.Body)
    else
        warn("[Analytics] Failed to get player stats:", result)
        return nil
    end
end

function Analytics.customEvent(userId, eventName, properties)
    return Analytics.track(userId, eventName, properties)
end

return Analytics
