--[[
  RobloxBackend Anti-Exploit SDK
  Version: 1.0.0
  
  Usage:
    local AntiExploit = require(game.ReplicatedStorage.AntiExploitSDK)
    AntiExploit.init("YOUR_API_KEY", "https://your-domain.com")
    
    -- Report an exploit
    AntiExploit.report(player.UserId, "SPEED_HACK", { speed = 100 })
    
    -- Check if player is banned
    local banned, reason = AntiExploit.checkBan(player.UserId)
]]

local HttpService = game:GetService("HttpService")

local AntiExploit = {}
AntiExploit.__index = AntiExploit

local API_KEY = ""
local BASE_URL = "https://dogan.works"

function AntiExploit.init(apiKey, baseUrl)
    API_KEY = apiKey
    BASE_URL = baseUrl or "https://dogan.works"
end

function AntiExploit.report(userId, exploitType, metadata)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/exploits/report",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                userId = tostring(userId),
                type = exploitType,
                metadata = metadata or {}
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.threatScore, data.actionTaken
    else
        warn("[AntiExploit] Failed to report exploit:", result)
        return false, 0, "ERROR"
    end
end

function AntiExploit.checkBan(userId)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/exploits/check-ban/" .. tostring(userId),
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        if data.banned then
            return true, data.reason, data.expiresAt, data.type
        else
            return false, nil, nil, nil
        end
    else
        warn("[AntiExploit] Failed to check ban:", result)
        return false, nil, nil, nil
    end
end

function AntiExploit.ban(userId, duration, reason, banType)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/exploits/ban",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                userId = tostring(userId),
                duration = duration, -- nil for permanent
                reason = reason,
                type = banType or "TEMPORARY"
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.ban
    else
        warn("[AntiExploit] Failed to ban player:", result)
        return false, nil
    end
end

function AntiExploit.unban(userId)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/exploits/ban?userId=" .. tostring(userId),
            Method = "DELETE",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success
    else
        warn("[AntiExploit] Failed to unban player:", result)
        return false
    end
end

-- Detection helpers
function AntiExploit.detectSpeedHack(player, maxSpeed)
    local character = player.Character
    if not character then return false end
    
    local humanoid = character:FindFirstChild("Humanoid")
    if not humanoid then return false end
    
    local hrp = character:FindFirstChild("HumanoidRootPart")
    if not hrp then return false end
    
    local velocity = hrp.AssemblyLinearVelocity
    local speed = math.sqrt(velocity.X^2 + velocity.Z^2)
    
    if speed > (maxSpeed or 50) then
        AntiExploit.report(player.UserId, "SPEED_HACK", { speed = speed })
        return true
    end
    
    return false
end

function AntiExploit.detectTeleport(player, lastPosition, maxDistance)
    local character = player.Character
    if not character then return false end

    local hrp = character:FindFirstChild("HumanoidRootPart")
    if not hrp then return false end

    if lastPosition then
        local distance = (hrp.Position - lastPosition).Magnitude
        if distance > (maxDistance or 100) then
            AntiExploit.report(player.UserId, "TELEPORT", { distance = distance })
            return true
        end
    end

    return false
end

function AntiExploit.getStats()
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/exploits/stats",
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)

    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data
    else
        warn("[AntiExploit] Failed to get stats:", result)
        return nil
    end
end

function AntiExploit.getReports(page, limit, filters)
    local url = BASE_URL .. "/api/v1/exploits/reports"
    local params = {}

    table.insert(params, "page=" .. (page or 1))
    table.insert(params, "limit=" .. (limit or 50))

    if filters then
        if filters.userId then
            table.insert(params, "userId=" .. tostring(filters.userId))
        end
        if filters.type then
            table.insert(params, "type=" .. filters.type)
        end
        if filters.startDate then
            table.insert(params, "startDate=" .. filters.startDate)
        end
        if filters.endDate then
            table.insert(params, "endDate=" .. filters.endDate)
        end
    end

    url = url .. "?" .. table.concat(params, "&")

    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = url,
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)

    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.reports, data.pagination
    else
        warn("[AntiExploit] Failed to get reports:", result)
        return {}, nil
    end
end

return AntiExploit
