--[[
  RobloxBackend Trading SDK
  Version: 1.0.0
  
  Usage:
    local Trading = require(game.ReplicatedStorage.TradingSDK)
    Trading.init("YOUR_API_KEY", "https://dogan.works")
    
    -- Create a trade
    local tradeId = Trading.createTrade(player1.UserId, player2.UserId)
    
    -- Add items
    Trading.addItems(tradeId, player1.UserId, {{ itemId = "sword", quantity = 1 }})
    
    -- Set ready
    Trading.setReady(tradeId, player1.UserId, true)
    
    -- Complete trade when both ready
    Trading.completeTrade(tradeId)
]]

local HttpService = game:GetService("HttpService")

local Trading = {}
Trading.__index = Trading

local API_KEY = ""
local BASE_URL = "https://dogan.works"

function Trading.init(apiKey, baseUrl)
    API_KEY = apiKey
    BASE_URL = baseUrl or "https://dogan.works"
end

function Trading.createTrade(player1Id, player2Id)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/create",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                player1Id = tostring(player1Id),
                player2Id = tostring(player2Id)
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.tradeId, data.expiresAt
    else
        warn("[Trading] Failed to create trade:", result)
        return nil, nil
    end
end

function Trading.addItems(tradeId, playerId, items)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/" .. tradeId .. "/add-item",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                playerId = tostring(playerId),
                items = items
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.currentTrade
    else
        warn("[Trading] Failed to add items:", result)
        return false, nil
    end
end

function Trading.setReady(tradeId, playerId, ready)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/" .. tradeId .. "/ready",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                playerId = tostring(playerId),
                ready = ready
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.bothReady
    else
        warn("[Trading] Failed to set ready:", result)
        return false, false
    end
end

function Trading.completeTrade(tradeId)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/" .. tradeId .. "/complete",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            }
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success, data.completedAt
    else
        warn("[Trading] Failed to complete trade:", result)
        return false, nil
    end
end

function Trading.cancelTrade(tradeId, playerId)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/" .. tradeId .. "/cancel",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                playerId = tostring(playerId)
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.success
    else
        warn("[Trading] Failed to cancel trade:", result)
        return false
    end
end

function Trading.verifyOwnership(playerId, items)
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/verify-ownership",
            Method = "POST",
            Headers = {
                ["Content-Type"] = "application/json",
                ["x-api-key"] = API_KEY
            },
            Body = HttpService:JSONEncode({
                playerId = tostring(playerId),
                items = items
            })
        })
    end)
    
    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.valid, data.missingItems
    else
        warn("[Trading] Failed to verify ownership:", result)
        return false, nil
    end
end

function Trading.getHistory(playerId, page, limit)
    local url = BASE_URL .. "/api/v1/trading/history/" .. tostring(playerId)
    url = url .. "?page=" .. (page or 1) .. "&limit=" .. (limit or 20)

    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = url,
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)

    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data.trades, data.pagination
    else
        warn("[Trading] Failed to get history:", result)
        return {}, nil
    end
end

function Trading.getStats()
    local success, result = pcall(function()
        return HttpService:RequestAsync({
            Url = BASE_URL .. "/api/v1/trading/stats",
            Method = "GET",
            Headers = {
                ["x-api-key"] = API_KEY
            }
        })
    end)

    if success and result.Success then
        local data = HttpService:JSONDecode(result.Body)
        return data
    else
        warn("[Trading] Failed to get stats:", result)
        return nil
    end
end

return Trading
